const express = require('express');
const cors = require('cors');
const sqlite3 = require('sqlite3').verbose();
const CryptoJS = require('crypto-js');
const path = require('path');

const config = process.env.NODE_ENV === 'production' 
  ? require('./config.production')
  : { port: 5000, databasePath: process.env.DATABASE_PATH || path.join(__dirname, 'database', 'notes.db'), corsOrigin: '*' };

const app = express();
const port = config.port;

app.use(cors({
  origin: config.corsOrigin
}));
app.use(express.json());

// Initialize database
const dbPath = config.databasePath;
const db = new sqlite3.Database(dbPath, (err) => {
  if (err) {
    console.error('Error connecting to database:', err);
  } else {
    console.log('Connected to database');
    db.run(`CREATE TABLE IF NOT EXISTS notes (
      id TEXT PRIMARY KEY,
      content TEXT,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      read_count INTEGER DEFAULT 0
    )`);
  }
});

// Create a new note
app.post('/api/notes', (req, res) => {
  const { content } = req.body;
  if (!content) {
    return res.status(400).json({ error: 'Content is required' });
  }

  const id = CryptoJS.lib.WordArray.random(16).toString();
  const encryptedContent = CryptoJS.AES.encrypt(content, id).toString();

  db.run('INSERT INTO notes (id, content) VALUES (?, ?)', [id, encryptedContent], (err) => {
    if (err) {
      return res.status(500).json({ error: 'Failed to create note' });
    }
    res.json({ id });
  });
});

// Read a note
app.get('/api/notes/:id', (req, res) => {
  const { id } = req.params;

  db.get('SELECT * FROM notes WHERE id = ?', [id], (err, note) => {
    if (err) {
      return res.status(500).json({ error: 'Failed to fetch note' });
    }
    if (!note) {
      return res.status(404).json({ error: 'Note not found' });
    }

    try {
      const decryptedContent = CryptoJS.AES.decrypt(note.content, id).toString(CryptoJS.enc.Utf8);
      
      // Update read count and delete if read
      db.run('DELETE FROM notes WHERE id = ?', [id], (err) => {
        if (err) {
          console.error('Error deleting note:', err);
        }
      });

      res.json({ content: decryptedContent });
    } catch (error) {
      res.status(400).json({ error: 'Invalid note ID' });
    }
  });
});

app.listen(port, () => {
  console.log(`Server running on port ${port}`);
});
