import React, { useState } from 'react';
import './App.css';

const themes = {
  blue: {
    '--background': '#e0f7fa',
    '--foreground': '#006064',
    '--primary': '#3b82f6',
    '--primary-foreground': '#ffffff',
    '--border': '#80deea',
    '--input': '#b2ebf2',
    '--muted-foreground': '#004d40'
  },
  orange: {
    '--background': '#fff3e0',
    '--foreground': '#e65100',
    '--primary': '#f97316',
    '--primary-foreground': '#ffffff',
    '--border': '#ffe0b2',
    '--input': '#ffcc80',
    '--muted-foreground': '#bf360c'
  },
  green: {
    '--background': '#e8f5e9',
    '--foreground': '#1b5e20',
    '--primary': '#10b981',
    '--primary-foreground': '#ffffff',
    '--border': '#a5d6a7',
    '--input': '#c8e6c9',
    '--muted-foreground': '#2e7d32'
  },
  zinc: {
    '--background': '#f5f5f5',
    '--foreground': '#212121',
    '--primary': '#71717a',
    '--primary-foreground': '#ffffff',
    '--border': '#bdbdbd',
    '--input': '#e0e0e0',
    '--muted-foreground': '#757575'
  }
};

function App() {
  const [noteContent, setNoteContent] = useState('');
  const [noteId, setNoteId] = useState('');
  const [showModal, setShowModal] = useState(false);
  const [noteLink, setNoteLink] = useState('');
  const [showNoteLink, setShowNoteLink] = useState(false);

  const applyTheme = (theme: keyof typeof themes) => {
    const root = document.documentElement;
    const themeColors = themes[theme];
    Object.entries(themeColors).forEach(([key, value]) => {
      root.style.setProperty(key, value);
    });
  };

  const createNote = async () => {
    if (!noteContent.trim()) return;

    try {
      const response = await fetch('http://localhost:5000/api/notes', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ content: noteContent })
      });

      const data = await response.json();
      if (data.id) {
        const link = `${window.location.origin}?note=${data.id}`;
        setNoteLink(link);
        setShowNoteLink(true);
        setNoteContent('');
      }
    } catch (error) {
      console.error('Error creating note:', error);
    }
  };

  const readNote = async () => {
    if (!noteId) return;
    
    try {
      const response = await fetch(`http://localhost:5000/api/notes/${noteId}`);
      const data = await response.json();
      
      if (data.content) {
        setNoteContent(data.content);
        setShowModal(false);
        setNoteId('');
      }
    } catch (error) {
      console.error('Error reading note:', error);
    }
  };

  // Set default theme
  React.useEffect(() => {
    applyTheme('blue');
    
    // Check for note ID in URL
    const params = new URLSearchParams(window.location.search);
    const noteId = params.get('note');
    if (noteId) {
      setNoteId(noteId);
      setShowModal(true);
    }
  }, []);

  return (
    <div className="min-h-screen flex flex-col items-center justify-center p-4" style={{ backgroundColor: 'var(--background)', color: 'var(--foreground)' }}>
      <div className="flex space-x-4 mb-8">
        {Object.keys(themes).map((theme) => (
          <button
            key={theme}
            onClick={() => applyTheme(theme as keyof typeof themes)}
            className="p-2 rounded-full"
            style={{ backgroundColor: themes[theme as keyof typeof themes]['--primary'] }}
            aria-label={`${theme} Theme`}
          >
            <div className="w-6 h-6 rounded-full" />
          </button>
        ))}
      </div>

      <div className="max-w-md w-full p-6 rounded-lg shadow-lg" style={{ backgroundColor: 'var(--card)', color: 'var(--card-foreground)' }}>
        <h1 className="text-3xl font-bold mb-4 text-center">Secure Note Sharing</h1>
        <textarea
          value={noteContent}
          onChange={(e) => setNoteContent(e.target.value)}
          className="w-full h-40 p-3 border rounded-lg mb-4"
          style={{ borderColor: 'var(--border)', backgroundColor: 'var(--input)', color: 'var(--foreground)' }}
          placeholder="Write your secure note here..."
        />
        <button
          onClick={createNote}
          className="w-full py-3 rounded-lg"
          style={{ backgroundColor: 'var(--primary)', color: 'var(--primary-foreground)' }}
        >
          Create Note
        </button>

        {showNoteLink && (
          <div className="mt-4">
            <p className="mb-2" style={{ color: 'var(--muted-foreground)' }}>Share this link to view the note:</p>
            <input
              value={noteLink}
              readOnly
              className="w-full p-2 border rounded-lg"
              style={{ borderColor: 'var(--border)', backgroundColor: 'var(--input)', color: 'var(--foreground)' }}
              onClick={(e) => (e.target as HTMLInputElement).select()}
            />
          </div>
        )}

        <div className="mt-4">
          <input
            value={noteId}
            onChange={(e) => setNoteId(e.target.value)}
            className="w-full p-2 border rounded-lg mb-2"
            style={{ borderColor: 'var(--border)', backgroundColor: 'var(--input)', color: 'var(--foreground)' }}
            placeholder="Enter note ID or URL..."
          />
          <button
            onClick={() => setShowModal(true)}
            className="w-full py-3 rounded-lg"
            style={{ backgroundColor: 'var(--primary)', color: 'var(--primary-foreground)' }}
          >
            Read Note
          </button>
        </div>
      </div>

      {showModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center">
          <div className="bg-white dark:bg-zinc-800 p-6 rounded-lg shadow-lg max-w-sm w-full">
            <h2 className="text-lg font-bold mb-4" style={{ color: 'var(--foreground)' }}>Confirm Action</h2>
            <p className="mb-4" style={{ color: 'var(--muted-foreground)' }}>
              Are you sure you want to read this note? This action cannot be undone.
            </p>
            <div className="flex justify-end space-x-2">
              <button
                onClick={() => setShowModal(false)}
                className="px-4 py-2 rounded-lg"
                style={{ backgroundColor: 'var(--destructive)', color: 'var(--destructive-foreground)' }}
              >
                Cancel
              </button>
              <button
                onClick={readNote}
                className="px-4 py-2 rounded-lg"
                style={{ backgroundColor: 'var(--primary)', color: 'var(--primary-foreground)' }}
              >
                Confirm
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}

export default App;
